<?php
/**
 * Waitlist class.
 *
 * @package woodmart
 */

namespace XTS\Modules\Waitlist;

use XTS\Admin\Modules\Options;
use XTS\Singleton;
use WC_Product;

/**
 * Waitlist class.
 */
class Main extends Singleton {
	/**
	 * DB_Storage instance.
	 *
	 * @var DB_Storage
	 */
	protected $db_storage;

	/**
	 * Init.
	 */
	public function init() {
		if ( ! woodmart_woocommerce_installed() ) {
			return;
		}

		$this->add_options();

		add_filter( 'woocommerce_settings_pages', array( $this, 'add_endpoint_option' ) );
		add_filter( 'woocommerce_get_query_vars', array( $this, 'add_endpoint' ) );

		if ( ! woodmart_get_opt( 'waitlist_enabled' ) ) {
			return;
		}

		$this->define_constants();
		$this->include_files();

		$this->db_storage = DB_Storage::get_instance();

		add_action( 'init', array( $this, 'custom_rewrite_rule' ) );

		add_action( 'before_delete_post', array( $this, 'remove_record_from_waitlist' ) );

		add_action( 'woodmart_remove_not_confirmed_emails', array( $this, 'remove_not_confirmed_emails' ) );

		if ( ! wp_next_scheduled( 'woodmart_remove_not_confirmed_emails' ) ) {
			wp_schedule_event( time(), apply_filters( 'woodmart_remove_not_confirmed_emails_time', 'daily' ), 'woodmart_remove_not_confirmed_emails' );
		}
	}

	/**
	 * Add options in theme settings.
	 */
	public function add_options() {
		Options::add_field(
			array(
				'id'          => 'waitlist_enabled',
				'name'        => esc_html__( 'فعالسازی لیست انتظار', 'woodmart' ),
				'hint' => '<video data-src="' . WOODMART_TOOLTIP_URL . 'waitlist_enabled.mp4" autoplay loop muted></video>',
				'description' => esc_html__( 'این گزینه را فعال کنید تا به مشتریان اجازه دهید به لیست انتظار برای محصولات بدون موجودی بپیوندند، و اطمینان حاصل کنید که وقتی کالاها دوباره در دسترس قرار می گیرند به آنها اطلاع داده می شود..', 'woodmart' ),
				'type'        => 'switcher',
				'section'     => 'waitlist_section',
				'default'     => '0',
				'on-text'     => esc_html__( 'Yes', 'woodmart' ),
				'off-text'    => esc_html__( 'No', 'woodmart' ),
				'priority'    => 10,
				'class'       => 'xts-preset-field-disabled',
			)
		);

		Options::add_field(
			array(
				'id'          => 'waitlist_for_loggined',
				'name'        => esc_html__( 'وورد برای مشاهده لیست انتظار', 'woodmart' ),
				'description' => esc_html__( 'ویژگی لیست انتظار را به کاربرانی که وارد سیستم شده اند محدود کنید، و اطمینان حاصل کنید که فقط مشتریان ثبت نام شده می توانند به لیست انتظار برای محصولات بدون موجودی بپیوندند..', 'woodmart' ),
				'type'        => 'switcher',
				'section'     => 'waitlist_section',
				'default'     => '0',
				'on-text'     => esc_html__( 'Yes', 'woodmart' ),
				'off-text'    => esc_html__( 'No', 'woodmart' ),
				'priority'    => 20,
				'class'       => 'xts-preset-field-disabled',
			)
		);

		Options::add_field(
			array(
				'id'          => 'waitlist_form_state',
				'name'        => esc_html__( 'حالت اولیه', 'woodmart' ),
				'description' => esc_html__( 'نمایش پیش فرض را برای ویژگی لیست انتظار انتخاب کنید: یا فرم پیوستن به لیست انتظار را نشان دهید یا وضعیت فعلی را نمایش دهید (الحاق شده یا نه).', 'woodmart' ),
				'type'        => 'buttons',
				'section'     => 'waitlist_section',
				'options'     => array(
					'always_open'   => array(
						'name'  => esc_html__( 'همیشه باز', 'woodmart' ),
						'value' => 'always_open',
					),
					'current_state' => array(
						'name'  => esc_html__( 'وضعیت فعلی', 'woodmart' ),
						'value' => 'current_state',
					),
				),
				'default'     => 'current_state',
				'priority'    => 30,
			)
		);

		Options::add_field(
			array(
				'id'          => 'waitlist_fragments_enable',
				'name'        => esc_html__( 'به روز رسانی قطعات را فعال کنید', 'woodmart' ),
				'description' => esc_html__( 'این تنظیم را فعال کنید تا مطمئن شوید که فرم لیست انتظار به درستی به‌روزرسانی می‌شود وقتی ذخیره پنهان فعال است، و اطلاعات فهرست انتظار دقیق در صفحه محصول حفظ می‌شود..', 'woodmart' ),
				'type'        => 'switcher',
				'section'     => 'waitlist_section',
				'default'     => '0',
				'on-text'     => esc_html__( 'Yes', 'woodmart' ),
				'off-text'    => esc_html__( 'No', 'woodmart' ),
				'priority'    => 40,
				'class'       => 'xts-preset-field-disabled',
				'requires'    => array(
					array(
						'key'     => 'waitlist_form_state',
						'compare' => 'equals',
						'value'   => 'always_open',
					),
				),
			)
		);

		Options::add_field(
			array(
				'id'          => 'waitlist_enable_privacy_checkbox',
				'name'        => esc_html__( 'فعال کردن کادر انتخاب خط مشی رازداری', 'woodmart' ),
				'hint' => '<video data-src="' . WOODMART_TOOLTIP_URL . 'waitlist_enable_privacy_checkbox.mp4" autoplay loop muted></video>',
				'description' => esc_html__( 'این تنظیم را فعال کنید تا از مشتریان بخواهید قبل از اینکه بتوانند به لیست انتظار برای محصولات بدون موجودی بپیوندند، با یک چک باکس با خط مشی رازداری شما موافقت کنند..', 'woodmart' ),
				'type'        => 'switcher',
				'section'     => 'waitlist_section',
				'default'     => '1',
				'on-text'     => esc_html__( 'Yes', 'woodmart' ),
				'off-text'    => esc_html__( 'No', 'woodmart' ),
				'priority'    => 50,
				'class'       => 'xts-preset-field-disabled',
			)
		);

		Options::add_field(
			array(
				'id'           => 'waitlist_privacy_checkbox_text',
				'name'         => esc_html__( 'متن چک باکس حریم خصوصی', 'woodmart' ),
				'description'  => esc_html__( 'متنی را مشخص کنید که در کنار کادر انتخاب خط مشی رازداری ظاهر می‌شود و مشتریان را در مورد خط‌مشی که باید قبل از پیوستن به فهرست انتظار با آن موافقت کنند، آگاه می‌کند. می‌توانید از کد کوتاه [شرایط] و [سیاست_مخصوصی] استفاده کنید', 'woodmart' ),
				'type'         => 'textarea',
				'wysiwyg'      => false,
				'section'      => 'waitlist_section',
				'empty_option' => true,
				'default'      => wp_kses( __('من <strong>[privacy_policy]</strong> را خوانده‌ام و می‌پذیرم', 'woodmart'), array( 'strong' => array() ) ),
				'priority'     => 60,
				'requires'     => array(
					array(
						'key'     => 'waitlist_enable_privacy_checkbox',
						'compare' => 'equals',
						'value'   => '1',
					),
				),
			)
		);
	}

	/**
	 * Add waiting list account endpoint option.
	 */
	public function add_endpoint_option( $settings ) {
		$offset       = array_search(
			'woocommerce_myaccount_payment_methods_endpoint',
			array_column(
				$settings,
				'id'
			),
			true
		) + 1;
		$first_part   = array_slice( $settings, 0, $offset, true );
		$last_part    = array_slice( $settings, $offset, null, true );
		$first_part[] = array(
			'title'    => esc_html__( 'لیست انتظار', 'woodmart' ),
			'desc'     => esc_html__( 'نقطه پایانی برای صفحه "حساب من &rarr; فهرست انتظار"..', 'woodmart' ),
			'id'       => 'woodmart_myaccount_waitlist_endpoint',
			'type'     => 'text',
			'default'  => 'waitlist',
			'desc_tip' => true,
		);
		$settings   = array_merge( $first_part, $last_part );

		return $settings;
	}

	/**
	 * Add waiting list account endpoint
	 */
	public function add_endpoint( $query_vars ) {
		$query_vars['waitlist'] = get_option( 'woodmart_myaccount_waitlist_endpoint', 'waitlist' );

		return $query_vars;
	}

	/**
	 * Define constants.
	 */
	private function define_constants() {
		if ( ! defined( 'XTS_WAITLIST_DIR' ) ) {
			define( 'XTS_WAITLIST_DIR', WOODMART_THEMEROOT . '/inc/integrations/woocommerce/modules/waitlist/' );
		}
	}

	/**
	 * Include files.
	 *
	 * @return void
	 */
	public function include_files() {
		$files = array(
			'class-db-storage',
			'class-emails',
			'class-admin',
			'class-frontend',
		);

		foreach ( $files as $file ) {
			$path = XTS_WAITLIST_DIR . $file . '.php';

			if ( file_exists( $path ) ) {
				require_once $path;
			}
		}
	}

	/**
	 * When the product is deleted, you should also delete this product from the waiting list for all users.
	 *
	 * @param int|string $post_id Post id.
	 */
	public function remove_record_from_waitlist( $post_id ) {
		$product = wc_get_product( $post_id );

		if ( ! $product instanceof WC_Product ) {
			return;
		}

		$this->db_storage->unsubscribe_by_product( $product );
	}

	/**
	 * Remove records if they have not been confirmed within 2 days after creating from the waiting list.
	 */
	public function remove_not_confirmed_emails() {
		$this->db_storage->remove_not_confirmed_emails();
	}

	/**
	 * Add rewrite rules for wishlist.
	 *
	 * @return void
	 */
	public function custom_rewrite_rule() {
		$myaccount_id = (int) get_option( 'woocommerce_myaccount_page_id' );
		$slug         = (string) get_post_field( 'post_name', $myaccount_id );

		if ( empty( $slug ) || ! array_key_exists( 'waitlist', WC()->query->query_vars ) ) {
			return;
		}

		$waitlist_endpoint = WC()->query->query_vars['waitlist'];

		add_rewrite_rule( '^' . $slug . '/' . $waitlist_endpoint . '/page/([^/]*)?', 'index.php?page_id=' . $myaccount_id . '&' . $waitlist_endpoint . '&paged=$matches[1]', 'top' );
	}
}

Main::get_instance();
